locals {
  bitcart_currency_btc_service = format("%s-currency-%s", local.bitcart_namespace, local.currency_btc)
  bitcart_namespace            = "bitcart"
  bitcart_service              = local.bitcart_namespace
  bitcart_volume_data          = "/data"

  currency_btc             = "btc"
  currency_btc_server_addr = format("%s.%s.svc.cluster.local:%d", local.electrum_service, local.electrum_namespace, 8000)

  lightning = true
  network   = "mainnet"
}

resource "kubernetes_namespace" "bitcart" {
  metadata {
	name   = local.bitcart_namespace
	labels = {
	  istio-injection = "enabled"
	}
  }
}

resource "kubernetes_config_map" "bitcart_currency_btc" {
  depends_on = [kubernetes_namespace.bitcart]

  metadata {
	name      = local.bitcart_currency_btc_service
	namespace = local.bitcart_namespace
  }

  data = {
	BTC_NETWORK          = local.network
	BTC_LIGHTNING        = local.lightning
	BTC_LIGHTNING_GOSSIP = false
	BTC_DEBUG            = false
	BTC_SERVER           = local.currency_btc_server_addr
  }
}

resource "kubernetes_persistent_volume" "bitcart" {
  depends_on = [kubernetes_namespace.bitcart]

  metadata {
	name = local.bitcart_service
  }

  spec {
	access_modes = ["ReadWriteOnce"]
	capacity     = {
	  storage = "2Gi"
	}

	persistent_volume_source {
	  host_path {
		path = format("/data/%s", local.bitcart_service)
	  }
	}

	storage_class_name = "standard"
  }
}

resource "kubernetes_persistent_volume_claim" "bitcart" {
  depends_on = [
	kubernetes_namespace.bitcart,
	kubernetes_persistent_volume.bitcart,
  ]

  metadata {
	name      = local.bitcart_service
	namespace = local.bitcart_namespace
  }

  spec {
	access_modes = ["ReadWriteOnce"]

	resources {
	  requests = {
		storage = "2Gi"
	  }
	}

	storage_class_name = "standard"
	volume_name        = local.bitcart_service
  }
}

resource "kubernetes_deployment" "bitcart_btc" {
  depends_on = [
	kubernetes_namespace.bitcart,
	kubernetes_config_map.bitcart_currency_btc,
	kubernetes_persistent_volume_claim.bitcart,
	kubernetes_deployment.electrum,
  ]

  metadata {
	name      = local.bitcart_service
	namespace = local.bitcart_namespace
	labels    = {
	  app = local.bitcart_service
	}
  }

  spec {
	replicas = 1

	selector {
	  match_labels = {
		app = local.bitcart_service
	  }
	}

	template {
	  metadata {
		annotations = {
		  "co.elastic.logs/enabled" = "false"
		  "sidecar.istio.io/inject" = "false"
		}
		labels = {
		  app = local.bitcart_service
		}
	  }

	  spec {
		container {
		  env_from {
			config_map_ref {
			  name = local.bitcart_currency_btc_service
			}
		  }

		  image             = format("bitcartcc/%s-%s:stable", local.bitcart_service, local.currency_btc)
		  image_pull_policy = "IfNotPresent"
		  name              = local.bitcart_currency_btc_service

		  port {
			name           = local.http_port_name
			container_port = 5000
		  }

		  resources {
			limits = {
			  cpu    = "0.1"
			  memory = "256M"
			}
			requests = {
			  cpu    = "0.05"
			  memory = "128M"
			}
		  }

		  #		  security_context {
		  #			privileged  = true
		  #			run_as_user = "0"
		  #		  }

		  volume_mount {
			mount_path = local.bitcart_volume_data
			name       = kubernetes_persistent_volume_claim.bitcart.metadata[0].name
		  }
		}

		volume {
		  name = kubernetes_persistent_volume_claim.bitcart.metadata[0].name
		  persistent_volume_claim {
			claim_name = local.bitcart_service
		  }
		}
	  }
	}
  }
}
