locals {
  electrum_namespace   = "electrum"
  electrum_service     = format("%s", local.electrum_namespace)
  electrum_volume_data = "/data"
}

resource "kubernetes_namespace" "electrum" {
  metadata {
	name = local.electrum_namespace
  }
}

resource "kubernetes_config_map" "electrum" {
  depends_on = [kubernetes_namespace.electrum]

  metadata {
	name      = local.electrum_service
	namespace = local.electrum_namespace
  }

  data = {
	COIN       = "Bitcoin"
	DAEMON_URL = join(",", var.electrum.daemon_urls)
	DB_DIRECTORY = local.electrum_volume_data
  }
}

resource "kubernetes_persistent_volume" "electrum" {
  depends_on = [kubernetes_namespace.electrum]

  metadata {
	name = local.electrum_service
  }

  spec {
	access_modes = ["ReadWriteOnce"]
	capacity     = {
	  storage = "1Gi"
	}

	persistent_volume_source {
	  host_path {
		path = format("/data/%s", local.electrum_service)
	  }
	}

	storage_class_name = "standard"
  }
}

resource "kubernetes_persistent_volume_claim" "electrum" {
  depends_on = [
	kubernetes_namespace.electrum,
	kubernetes_persistent_volume.electrum,
  ]

  metadata {
	name      = local.electrum_service
	namespace = local.electrum_namespace
  }

  spec {
	access_modes = ["ReadWriteOnce"]

	resources {
	  requests = {
		storage = "1Gi"
	  }
	}

	storage_class_name = "standard"
	volume_name        = local.electrum_service
  }
}

resource "kubernetes_deployment" "electrum" {
  depends_on = [
	kubernetes_namespace.electrum,
	kubernetes_config_map.electrum,
  ]

  metadata {
	name      = local.electrum_service
	namespace = local.electrum_namespace
	labels    = {
	  app = local.electrum_service
	}
  }

  spec {
	replicas = 1

	selector {
	  match_labels = {
		app = local.electrum_service
	  }
	}

	template {
	  metadata {
		labels = {
		  app = local.electrum_service
		}
	  }

	  spec {
		container {
		  env_from {
			config_map_ref {
			  name = local.electrum_service
			}
		  }

		  image             = "lukechilds/electrumx:v1.16.0"
		  image_pull_policy = "IfNotPresent"
		  name              = local.electrum_service

		  port {
			name           = local.http_port_name
			container_port = 8000
		  }

		  resources {
			limits = {
			  cpu    = "0.1"
			  memory = "256M"
			}
			requests = {
			  cpu    = "0.05"
			  memory = "128M"
			}
		  }

		  volume_mount {
			mount_path = local.electrum_volume_data
			name       = kubernetes_persistent_volume_claim.electrum.metadata[0].name
		  }
		}

		volume {
		  name = kubernetes_persistent_volume_claim.electrum.metadata[0].name
		  persistent_volume_claim {
			claim_name = local.electrum_service
		  }
		}
	  }
	}
  }
}
